## This script draws on ‘Simulation model/Cohort simulation model - source for n and wealth.R’ to generate 
## population, wealth, and wealth transfer sizes at each year, each cohort and for each scenario 
## both with and without the existence of future wealth transfers.

# Prelims - remem to update paths as needed -----------------------------------------------------------------

rm(list=ls())
gc()

memory.limit(120000)

today_date <- Sys.Date() 

## ensure packages are loaded
source("./R scripts/Master package loading.R")

## ** UPDATE ME** source file for calcs of n and wealth
model_source <- "./R scripts/Simulation model/Cohort simulation model - source for n and wealth.R"

## ** UPDATE ME** starting cohorts data
starting_cohorts_path <- "./Input data/starting_cohorts_aiwbh_p.qs"


## reset indicators for specific scenarios. Set to 1 only in that particular scenario, which will activate different code in source
w_bequests_dont_affect_homeownership <- 0
w_2b_wealthy_save_more <- 0
w_2c_anticipation_effect <- 0
w_2f_housing_for_gifts <- 0
w_bequests <- -1 ## 0 and 1 trigger diff parts of code - set these to -1
w_gifts <- -1


# Additional files for running model --------------------------------------

## discount rate for wealth = income growth projections
awe <- qread("./Input data/AWE_projections.qs") %>%
  select(year, awe_growth_factor_2018) %>%
  filter(year<=2050)

## parent mortality rates = bequest receipt prob
bequest_receipt_prob <- qread("./Input data/bequest_receipt_prob_aiwbh_yp.qs") 

## partnered status
partnered_status <- qread("./Input data/partnered_a.qs") %>% 
  rename(parent_age_grp = age_grp)

## distributions from age-specific bequest pools to prev wealth qtile
## Note age_wlth_grp3=0 is for age groups <30 or >90 - where bequest distributed evenly
age_wlth_beq_dist <- qread("./Input data/age_wlth_beq_dist.qs")

## starting cohorts
starting_cohorts <- qread(starting_cohorts_path) %>% 
  rename(housing_assets = housing_assets_av_smooth,
         super_assets = super_assets_av_smooth,
         other_assets = other_assets_av_smooth,
         housing_debt = housing_debt_av_smooth,
         
         wages_inc = wages_av_smooth,
         other_inc = other_inc_av_smooth) %>% 
  mutate(starting_cohort = paste0("SA", age_grp, " SI", total_inc_qtile, " SW", total_wealth_qtile, " SB", bequest_received, " SH", homeowner))

## this version contains n and wealth vars only
starting_cohorts_n_wealth <- starting_cohorts %>% 
  select(starting_cohort, n, matches("parent_age0_n|_assets|_debt")) %>% 
  mutate(year=0)


## convergence in rates of return
converge_rate <- 0.04
converge_years <- 10



# Base scenario -----------------------------------------------------------

# all rates of returns converge to 4% over 10 years 

## read in year_list_0 for convergence model
year_list_0 <- qread("./Output data/year_list_0_returns_converge.qs")

## With transfers
start_time <- Sys.time()

w_transfers <- 1
source(model_source)

runtime <- Sys.time()-start_time
runtime

qsave(year_list, paste0("./Output data/sim_results_returns_converge_", today_date, ".qs"))


## Without transfers
start_time <- Sys.time()

w_transfers <- 0
source(model_source)

runtime <- Sys.time()-start_time
runtime

qsave(year_list, paste0("./Output data/sim_results_returns_converge_no_transfer_", today_date, ".qs"))




# Only inheritances, no gifts ---------------------------------------------

## read in year_list_0 for convergence model
year_list_0 <- qread("./Output data/year_list_0_returns_converge.qs")

## With inheritances only
start_time <- Sys.time()

w_transfers <- -1 ## set to -1 so it won't trigger anything in the code
w_bequests <- 1
w_gifts <- 0
source(model_source)

runtime <- Sys.time()-start_time
runtime

qsave(year_list, paste0("./Output data/sim_results_bequests_only_", today_date, ".qs"))

w_bequests <- -1
w_gifts <- -1



# Only gifts, no inheritances ---------------------------------------------

## read in year_list_0 for convergence model
year_list_0 <- qread("./Output data/year_list_0_returns_converge.qs")

## With gifts only
start_time <- Sys.time()

w_transfers <- -1 ## set to -1 so it won't trigger anything in the code
w_bequests <- 0
w_gifts <- 1
source(model_source)

runtime <- Sys.time()-start_time
runtime

qsave(year_list, paste0("./Output data/sim_results_gifts_only_", today_date, ".qs"))

## reset w_bequests and w_gifts
w_bequests <- -1
w_gifts <- -1



# RATE OF RETURN ADJUSTMENTS ----------------------------------------------

# 1a: Historical rates --------------------------------------------------------------

## Historical rates of return used (housing 7 per cent, superannuation 5.35–6.00 per cent, other assets 3.3–5.0 per cent).

## read in year_list_0 for historical model
year_list_0 <- qread("./Output data/year_list_0_historic.qs")


## with transfers
start_time <- Sys.time()

w_transfers <- 1
source(model_source)

runtime <- Sys.time()-start_time
runtime

qsave(year_list, paste0("./Output data/sim_results_historic_", today_date, ".qs"))


## without transfers
start_time <- Sys.time()

w_transfers <- 0
source(model_source)

runtime <- Sys.time()-start_time
runtime

qsave(year_list, paste0("./Output data/sim_results_historic_no_transfer_", today_date, ".qs"))




# 1b: High returns ---------------------------------------------------------

## All rates of return on assets converge to 5.6 per cent. (40% higher than convergence case)

## read in year_list_0 for base model
year_list_0 <- qread("./Output data/year_list_0_historic.qs") %>% 
  ## adjust rates of return 
  lapply(., function(x) {
    x %>% mutate(housing_asset_return_param = ifelse(year %in% c(1:converge_years),
                                                     housing_asset_return_param-(housing_asset_return_param-(converge_rate*1.4))/(converge_years-1)*(year-1),
                                                     converge_rate*1.4), 
                 housing_asset_saving_param = ifelse(housing_asset_saving_param>0,
                                                     ## reduce value of new purchases by degree of difference between this scenario and historic scenario returns
                                                     housing_asset_saving_param* housing_asset_return_param/0.07,
                                                     housing_asset_saving_param),
                 super_asset_return_param = ifelse(year %in% c(1:converge_years),
                                                   super_asset_return_param-(super_asset_return_param-(converge_rate*1.4))/(converge_years-1)*(year-1),
                                                   (converge_rate*1.4)), 
                 other_asset_return_param = ifelse(year %in% c(1:converge_years),
                                                   other_asset_return_param-(other_asset_return_param-(converge_rate*1.4))/(converge_years-1)*(year-1),
                                                   (converge_rate*1.4))
    )
  })

## with transfers
start_time <- Sys.time()

w_transfers <- 1
source(model_source)

runtime <- Sys.time()-start_time
runtime

qsave(year_list, paste0("./Output data/sim_results_high_growth_", today_date, ".qs"))


## Without transfers
start_time <- Sys.time()

w_transfers <- 0
source(model_source)

runtime <- Sys.time()-start_time
runtime

qsave(year_list, paste0("./Output data/sim_results_high_growth_no_transfer_", today_date, ".qs"))




# 1c: Low returns -----------------------------------------------------

## All rates of return on assets converge to 2.4 per cent. Assume all rates of return converge to 40% lower 


## read in year_list_0 for historic model
year_list_0 <- qread("./Output data/year_list_0_historic.qs") %>% 
  ## adjust rates of return 
  lapply(., function(x) {
    x %>% mutate(housing_asset_return_param = ifelse(year %in% c(1:converge_years),
                                                     housing_asset_return_param-(housing_asset_return_param-(converge_rate*0.6))/(converge_years-1)*(year-1),
                                                     converge_rate*0.6), 
                 housing_asset_saving_param = ifelse(housing_asset_saving_param>0,
                                                     ## reduce value of new purchases by degree of difference between this scenario and historic scenario returns
                                                     housing_asset_saving_param* housing_asset_return_param/0.07,
                                                     housing_asset_saving_param),
                 super_asset_return_param = ifelse(year %in% c(1:converge_years),
                                                   super_asset_return_param-(super_asset_return_param-(converge_rate*0.6))/(converge_years-1)*(year-1),
                                                   (converge_rate*0.6)), 
                 other_asset_return_param = ifelse(year %in% c(1:converge_years),
                                                   other_asset_return_param-(other_asset_return_param-(converge_rate*0.6))/(converge_years-1)*(year-1),
                                                   (converge_rate*0.6))
    )
  })



## with transfers
start_time <- Sys.time()

w_transfers <- 1
source(model_source)

runtime <- Sys.time()-start_time
runtime

qsave(year_list, paste0("./Output data/sim_results_low_growth_", today_date, ".qs"))


## Without transfers
start_time <- Sys.time()

w_transfers <- 0
source(model_source)

runtime <- Sys.time()-start_time
runtime

qsave(year_list, paste0("./Output data/sim_results_low_growth_no_transfer_", today_date, ".qs"))


# 1d: Zero returns -----------------------------------------

## All rates of return on assets are zero. 

## read in year_list_0 for historic model
year_list_0 <- qread("./Output data/year_list_0_historic.qs") %>% 
  ## adjust rate of return 
  lapply(., function(x) {
    x %>% mutate(housing_asset_return_param = 0,
                 other_asset_return_param = 0,
                 super_asset_return_param = 0,
                 housing_asset_saving_param = ifelse(housing_asset_saving_param>0,
                                                     ## reduce value of new purchases by degree of difference between this scenario and base scenario returns
                                                     housing_asset_saving_param*housing_asset_return_param/0.07,
                                                     housing_asset_saving_param)
    )
  })


## With transfers
start_time <- Sys.time()

w_transfers <- 1
source(model_source)

runtime <- Sys.time()-start_time
runtime

qsave(year_list, paste0("./Output data/sim_results_zero_returns_", today_date, ".qs"))



## without transfers
start_time <- Sys.time()

w_transfers <- 0
source(model_source)

runtime <- Sys.time()-start_time
runtime

qsave(year_list, paste0("./Output data/sim_results_zero_returns_no_transfer_", today_date, ".qs"))



# 1e: Housing return 5% (housing counterfactual used in chapter) ---------------------

## Counterfactual housing return assumption — housing rate of return converges to 5 per cent instead of 4 per cent.


## read in year_list_0 for historic model
year_list_0 <- qread("./Output data/year_list_0_historic.qs") %>% 
  ## adjust rate of return 
  lapply(., function(x) {
    x %>% mutate(housing_asset_return_param = ifelse(year %in% c(1:converge_years),
                                                     housing_asset_return_param-(housing_asset_return_param-(converge_rate+0.01))/(converge_years-1)*(year-1),
                                                     (converge_rate+0.01)), 
                 housing_asset_saving_param = ifelse(housing_asset_saving_param>0,
                                                     ##  reduce value of new purchases by degree of difference between this scenario and historic scenario returns
                                                     housing_asset_saving_param* housing_asset_return_param/0.07,
                                                     housing_asset_saving_param),
                 super_asset_return_param = ifelse(year %in% c(1:converge_years),
                                                   super_asset_return_param-(super_asset_return_param-converge_rate)/(converge_years-1)*(year-1),
                                                   converge_rate), 
                 other_asset_return_param = ifelse(year %in% c(1:converge_years),
                                                   other_asset_return_param-(other_asset_return_param-converge_rate)/(converge_years-1)*(year-1),
                                                   converge_rate)
    )
  })

## with transfers
start_time <- Sys.time()

w_transfers <- 1
source(model_source)

runtime <- Sys.time()-start_time
runtime

qsave(year_list, paste0("./Output data/sim_results_housing_cf_", today_date, ".qs"))


## Without transfers
start_time <- Sys.time()

w_transfers <- 0
source(model_source)

runtime <- Sys.time()-start_time
runtime

qsave(year_list, paste0("./Output data/sim_results_housing_cf_no_transfer_", today_date, ".qs"))



# Chapter scenario: housing returns at 7%  (for in text result in chapter) ---------------------

## read in year_list_0 for historic model
year_list_0 <- qread("./Output data/year_list_0_historic.qs") %>% 
  ## adjust rate of return for non-housing to match converge scenario
  lapply(., function(x) {
    x %>% mutate(                 super_asset_return_param = ifelse(year %in% c(1:converge_years),
                                                                    super_asset_return_param-(super_asset_return_param-converge_rate)/(converge_years-1)*(year-1),
                                                                    converge_rate), 
                                  other_asset_return_param = ifelse(year %in% c(1:converge_years),
                                                                    other_asset_return_param-(other_asset_return_param-converge_rate)/(converge_years-1)*(year-1),
                                                                    converge_rate)
    )
  })


## With transfers
start_time <- Sys.time()

w_transfers <- 1
source(model_source)

runtime <- Sys.time()-start_time
runtime

qsave(year_list, paste0("./Output data/sim_results_historic_housing_", today_date, ".qs"))

## Without transfers
start_time <- Sys.time()

w_transfers <- 0
source(model_source)

runtime <- Sys.time()-start_time
runtime

qsave(year_list, paste0("./Output data/sim_results_historic_housing_no_transfer_", today_date, ".qs"))



# Chapter scenario - housing returns converge to 3 per cent (used for "fan" charts) ---------------------

## housing LR rates of return are 1% LOWER than base convergence model

## read in year_list_0 for historic model
year_list_0 <- qread("./Output data/year_list_0_historic.qs") %>% 
  ## adjust rate of return 
  lapply(., function(x) {
    x %>% mutate(housing_asset_return_param = ifelse(year %in% c(1:converge_years),
                                                     housing_asset_return_param-(housing_asset_return_param-(converge_rate-0.01))/(converge_years-1)*(year-1),
                                                     (converge_rate-0.01)), 
                 housing_asset_saving_param = ifelse(housing_asset_saving_param>0,
                                                     ## reduce value of new purchases by degree of difference between this scenario and historic scenario returns
                                                     housing_asset_saving_param* housing_asset_return_param/0.07,
                                                     housing_asset_saving_param),
                 super_asset_return_param = ifelse(year %in% c(1:converge_years),
                                                   super_asset_return_param-(super_asset_return_param-converge_rate)/(converge_years-1)*(year-1),
                                                   converge_rate), 
                 other_asset_return_param = ifelse(year %in% c(1:converge_years),
                                                   other_asset_return_param-(other_asset_return_param-converge_rate)/(converge_years-1)*(year-1),
                                                   converge_rate)
    )
  })

## with transfers
start_time <- Sys.time()

w_transfers <- 1
source(model_source)

runtime <- Sys.time()-start_time
runtime

qsave(year_list, paste0("./Output data/sim_results_housing_cf2_", today_date, ".qs"))


## Without transfers
start_time <- Sys.time()

w_transfers <- 0
source(model_source)

runtime <- Sys.time()-start_time
runtime

qsave(year_list, paste0("./Output data/sim_results_housing_cf2_no_transfer_", today_date, ".qs"))





# WEALTH TRANSFER ADJUSTMENTS ------------------------------


# 2a: Wealthy receive larger inheritances ---------------------------------------------

## Inheritance recipients aged 30–89 in the top third of the wealth distribution for their age group receive
## 65 per cent of the pool of inheritances for their age group, the middle third receive 30 per cent, and the 
## bottom third receive 5 per cent. It is also assumed that receiving an inheritance does not lead to an increased 
## probability of homeownership.

age_wlth_beq_dist <- tibble(
  prev_age_wlth_grp3 = c(1,2,3,0),
  age_wlth_bequest_dist = c(0.05, 0.30, 0.65, 1)
)


## set indicator for source to remove bequest effect on housing
w_bequests_dont_affect_homeownership <- 1


## read in year_list_0 for this scenario
year_list_0 <- qread("./Output data/year_list_0_returns_converge.qs") 


## with transfers
start_time <- Sys.time()

w_transfers <- 1
source(model_source)

runtime <- Sys.time()-start_time
runtime

qsave(year_list, paste0("./Output data/sim_results_low_wealth_low_beq_dist_", today_date, ".qs"))


## Without transfers - same as base without transfers
base_without_transfers <- qread(paste0("./Output data/sim_results_returns_converge_no_transfer_", today_date, ".qs"))
qsave(base_without_transfers, paste0("./Output data/sim_results_low_wealth_low_beq_dist_no_transfer_", today_date, ".qs"))



## reset dist at the end of scenario
age_wlth_beq_dist <- qread("./Input data/age_wlth_beq_dist.qs")

## reset indicator for source to merge in new bequest saving rates
w_bequests_dont_affect_homeownership <- 0



# 2b: Wealthy save more of inheritance --------------------------------------------

## Inheritance recipients aged 30–89 in the top third of the wealth distribution for their age group save 40 per cent
## of their received inheritance in other assets upon receipt, the middle third saves 20 per cent, and the bottom third saves 5 per cent.

## read in year_list_0 for this scenario
year_list_0 <- qread("./Output data/year_list_0_returns_converge.qs") 

## read in bequest saving rates for low wealth bequest depletion scenario. Will be merged in in source model
bequest_saving_age_wlth_grp3 <- qread("./Input data/bequest_other_asset_saving_grp3.qs")


## set indicator for source to merge in new bequest saving rates
w_2b_wealthy_save_more <- 1

## with transfers
start_time <- Sys.time()

w_transfers <- 1
source(model_source)

runtime <- Sys.time()-start_time
runtime

qsave(year_list, paste0("./Output data/sim_results_low_wealth_beq_depletion_", today_date, ".qs"))


## Without transfers - same as base without transfers
base_without_transfers <- qread(paste0("./Output data/sim_results_returns_converge_no_transfer_", today_date, ".qs"))
qsave(base_without_transfers, paste0("./Output data/sim_results_low_wealth_beq_depletion_no_transfer_", today_date, ".qs"))


## reset indicator for source to merge in new bequest saving rates
w_2b_wealthy_save_more <- 0




# 2c: Less wealthy save less before receiving inheritance ---------------------------------

## Cohorts aged 30–89 that have not yet received an inheritance and are in the top third of the wealth distribution 
## for their age group have a rate of saving on income in other assets that is 10 per cent lower than under the base scenario, 
## the middle third have a saving rate that is 20 per cent lower, and the bottom third have a saving rate that is 40 per cent lower. 
## Saving rates on income in other assets revert to base scenario rates upon receiving an inheritance.


## read in year_list_0 for this scenario
year_list_0 <- qread("./Output data/year_list_0_returns_converge.qs") 

w_2c_anticipation_effect <- 1


## with transfers
start_time <- Sys.time()

w_transfers <- 1
source(model_source)

runtime <- Sys.time()-start_time
runtime

qsave(year_list, paste0("./Output data/sim_results_low_wealth_low_saving_b4_", today_date, ".qs"))



## Without transfers - same as base without transfers
base_without_transfers <- qread(paste0("./Output data/sim_results_returns_converge_no_transfer_", today_date, ".qs"))
qsave(base_without_transfers, paste0("./Output data/sim_results_low_wealth_low_saving_b4_no_transfer_", today_date, ".qs"))


## reset
w_2c_anticipation_effect <- 0



# 2d: Older cohorts save more of inheritance --------------------------------------------

## Cohorts aged 55 and over who receive an inheritance save their inheritance in other assets 
## at double the saving rate of income in other assets.

## read in year_list_0 for this scenario
year_list_0 <- qread("./Output data/year_list_0_increase_bequest_saving.qs")

## with transfers
start_time <- Sys.time()

w_transfers <- 1
source(model_source)

runtime <- Sys.time()-start_time
runtime

qsave(year_list, paste0("./Output data/sim_results_increase_bequest_saving_", today_date, ".qs"))


## Without transfers - same as base without transfers
base_without_transfers <- qread(paste0("./Output data/sim_results_returns_converge_no_transfer_", today_date, ".qs"))
qsave(base_without_transfers, paste0("./Output data/sim_results_increase_bequest_saving_no_transfer_", today_date, ".qs"))



# 2e: Save 80 per cent of inheritances ------------------------------------------

## All cohorts save 80 per cent of their inheritance in other assets.
## Receiving an inheritance does not lead to an increased probability of homeownership.

## read in year_list_0 for model
year_list_0 <- qread("./Output data/year_list_0_returns_converge.qs") %>% 
  ## adjust saving
  lapply(., function(x) {
    x %>% mutate(bequest_other_asset_saving_param = 0.8)
    
  })

## turn off bequest effect on housing
w_bequests_dont_affect_homeownership <- 1


## with transfers
start_time <- Sys.time()

w_transfers <- 1
source(model_source)

runtime <- Sys.time()-start_time
runtime

qsave(year_list, paste0("./Output data/sim_results_bequest_saving_most_", today_date, ".qs"))


## Without transfers - same as base without transfers
base_without_transfers <- qread(paste0("./Output data/sim_results_returns_converge_no_transfer_", today_date, ".qs"))
qsave(base_without_transfers, paste0("./Output data/sim_results_bequest_saving_most_no_transfer_", today_date, ".qs"))


w_bequests_dont_affect_homeownership <- 0



# 2f: Housing for gifts -------------------------------------------------------

## Cohorts aged 60–79 withdraw 0.3 per cent from housing assets each year to give as gifts 
## (similar to assumed housing drawdowns for the age group). (No assumptions were made on the use of gifts to purchase housing.)

## read in year_list_0 for this scenario
year_list_0 <- qread("./Output data/year_list_0_returns_converge.qs") 


w_2f_housing_for_gifts <- 1

housing_for_gifts_rate <- 0.003


## with transfers
start_time <- Sys.time()

w_transfers <- 1
source(model_source)

runtime <- Sys.time()-start_time
runtime

qsave(year_list, paste0("./Output data/sim_results_housing_for_gifts_", today_date, ".qs"))



## Without transfers - same as base without transfers
base_without_transfers <- qread(paste0("./Output data/sim_results_returns_converge_no_transfer_", today_date, ".qs"))
qsave(base_without_transfers, paste0("./Output data/sim_results_housing_for_gifts_no_transfer_", today_date, ".qs"))


w_2f_housing_for_gifts <- 0




# HOUSING ADJUSTMENTS -------------------------------------------


# 3a: Alternate method for estimating probability of homeownership -------------

## Longitudinal panel HILDA data (rather than cross-sectional data across age groups) used to estimate probability 
## of homeownership. This resulted in younger cohorts taking longer to get into homeownership but overall higher 
## homeownership rates by the time cohorts reach age 50.


## read in year_list_0 for this scenario
year_list_0 <- qread("./Output data/year_list_0_alt_home_prob.qs") 


## with transfers
start_time <- Sys.time()

w_transfers <- 1
source(model_source)

runtime <- Sys.time()-start_time
runtime

qsave(year_list, paste0("./Output data/sim_results_alt_home_prob_", today_date, ".qs"))


## Without transfers
start_time <- Sys.time()

w_transfers <- 0
source(model_source)

runtime <- Sys.time()-start_time
runtime

qsave(year_list, paste0("./Output data/sim_results_alt_home_prob_no_transfer_", today_date, ".qs"))



# 3b: Housing debt to asset ratio on purchase is 83 per cent ----------------------------------------------

## Housing debt to asset ratio on housing purchases assumed to be 83 per cent across all cohorts (higher than base case), 
## based on RBA estimate for indebted first home buyer households (which excluded those with no debt) (Simon and Stone 2017, p. 14). 

## read in year_list_0 for base model
year_list_0 <- qread("./Output data/year_list_0_returns_converge.qs") %>% 
  ## adjust debt to asset ratio
  lapply(., function(x) {
    x %>% mutate(housing_debt_to_asset_param = 0.83)
  })


## with transfers
start_time <- Sys.time()

w_transfers <- 1
source(model_source)

runtime <- Sys.time()-start_time
runtime

qsave(year_list, paste0("./Output data/sim_results_housing_debt83_", today_date, ".qs"))


## Without transfers
start_time <- Sys.time()

w_transfers <- 0
source(model_source)

runtime <- Sys.time()-start_time
runtime

qsave(year_list, paste0("./Output data/sim_results_housing_debt83_no_transfer_", today_date, ".qs"))



# 3c: Same housing entry price across all income groups ------------------------------------------

## All cohorts in the same income quintile enter property market with same housing entry price as top income quintile in the base scenario.

## read in year_list_0 for this scenario
year_list_0 <- qread("./Output data/year_list_0_same_housing_entry_price.qs") 


## with transfers
start_time <- Sys.time()

w_transfers <- 1
source(model_source)

runtime <- Sys.time()-start_time
runtime

qsave(year_list, paste0("./Output data/sim_results_same_housing_entry_price_", today_date, ".qs"))


## Without transfers
start_time <- Sys.time()

w_transfers <- 0
source(model_source)

runtime <- Sys.time()-start_time
runtime

qsave(year_list, paste0("./Output data/sim_results_same_housing_entry_price_no_transfer_", today_date, ".qs"))



# 3d: Housing entry prices increase over time ------------------------------------------------

## Housing entry prices increase by 2 per cent more than base scenario housing entry price each year. 
## Housing returns are at historical rates.

## read in year_list_0 for this scenario
year_list_0 <- qread("./Output data/year_list_0_higher_housing_entry_price.qs") 


## with transfers
start_time <- Sys.time()

w_transfers <- 1
source(model_source)

runtime <- Sys.time()-start_time
runtime

qsave(year_list, paste0("./Output data/sim_results_higher_housing_entry_price_", today_date, ".qs"))


## Without transfers
start_time <- Sys.time()

w_transfers <- 0
source(model_source)

runtime <- Sys.time()-start_time
runtime

qsave(year_list, paste0("./Output data/sim_results_higher_housing_entry_price_no_transfer_", today_date, ".qs"))



# 3e: Housing affordability declines over time ---------------------------------

## Probability of entry into homeownership falls by 2 per cent each year for everyone but the highest income quintile. 
## Entry prices for housing increase 2 per cent faster each year. Housing returns are at historical rates.

## read in year_list_0 for this scenario
year_list_0 <- qread("./Output data/year_list_0_declining_house_afford.qs") 

## with transfers
start_time <- Sys.time()

w_transfers <- 1
source(model_source)

runtime <- Sys.time()-start_time
runtime

qsave(year_list, paste0("./Output data/sim_results_declining_house_afford_", today_date, ".qs"))


## Without transfers
start_time <- Sys.time()

w_transfers <- 0
source(model_source)

runtime <- Sys.time()-start_time
runtime

qsave(year_list, paste0("./Output data/sim_results_declining_house_afford_no_transfer_", today_date, ".qs"))



# 3f: Lower housing demand --------------------------------------------------

## Probability of entry into homeownership falls by 5 per cent each year in first 5 years and then stays stable. 
## Housing entry prices grow 2 per cent slower than base scenario each year in first 5 years and then 0.5 per cent 
## slower each year thereafter. Housing returns fall from 7 to 5 per cent within 5 years.


## read in year_list_0 for this scenario
year_list_0 <- qread("./Output data/year_list_0_lower_housing_demand.qs") 

## with transfers
start_time <- Sys.time()

w_transfers <- 1
source(model_source)

runtime <- Sys.time()-start_time
runtime

qsave(year_list, paste0("./Output data/sim_results_lower_housing_demand_", today_date, ".qs"))


## Without transfers
start_time <- Sys.time()

w_transfers <- 0
source(model_source)

runtime <- Sys.time()-start_time
runtime

qsave(year_list, paste0("./Output data/sim_results_lower_housing_demand_no_transfer_", today_date, ".qs"))




# OTHER ADJUSTMENTS ------------------------------------------

# 4a: Alternate method for estimating drawdowns in other assets  --------

## Cross-sectional HILDA data across age groups (rather than balanced longitudinal panel data) used to estimate 
## difference between observed change in wealth and estimated change (given assumed rates of return and saving). 
## Results in similar patterns of drawdowns but some higher rates, particularly for older homeowners.

year_list_0 <- qread("./Output data/year_list_0_alt_other_drawdown.qs") 


## with transfers
start_time <- Sys.time()

w_transfers <- 1
source(model_source)

runtime <- Sys.time()-start_time
runtime

qsave(year_list, paste0("./Output data/sim_results_alt_other_drawdown_", today_date, ".qs"))


## Without transfers
start_time <- Sys.time()

w_transfers <- 0
source(model_source)

runtime <- Sys.time()-start_time
runtime

qsave(year_list, paste0("./Output data/sim_results_alt_other_drawdown_no_transfer_", today_date, ".qs"))



# 4b: Older cohorts’ drawdowns from wealth increase over time ---------------------------

## Drawdown rates on other assets for cohorts aged 65+ increase by 5 per cent of base scenario rates each year 
## for next 10 years and then stay stable (new rates range from 6–10 per cent). Drawdown rates on housing for 
## cohorts aged 70+ increase by 15 per cent of base scenario rates each year for next 10 years and then stay 
## stable (new rates range from 0.7–1.3 per cent).

## read in year_list_0 for this scenario
year_list_0 <- qread("./Output data/year_list_0_increase_drawdowns_larger.qs") 


## with transfers
start_time <- Sys.time()

w_transfers <- 1
source(model_source)

runtime <- Sys.time()-start_time
runtime

qsave(year_list, paste0("./Output data/sim_results_increase_drawdowns_larger_", today_date, ".qs"))


## Without transfers
start_time <- Sys.time()

w_transfers <- 0
source(model_source)

runtime <- Sys.time()-start_time
runtime

qsave(year_list, paste0("./Output data/sim_results_increase_drawdowns_larger_no_transfer_", today_date, ".qs"))



# 4c: All wealth passed on to younger generations at death ----------------------------------------------------

## Edit probability of having a partner at death to 0
partnered_status <- qread("./Input data/partnered_a.qs") %>% 
  rename(parent_age_grp = age_grp) %>% 
  mutate(partnered_pc=0)

## read in year_list_0 for base model
year_list_0 <- qread("./Output data/year_list_0_returns_converge.qs")


## with transfers
start_time <- Sys.time()

w_transfers <- 1
source(model_source)

runtime <- Sys.time()-start_time
runtime

qsave(year_list, paste0("./Output data/sim_results_all_wealth_passed_", today_date, ".qs"))


## Without transfers
start_time <- Sys.time()

w_transfers <- 0
source(model_source)

runtime <- Sys.time()-start_time
runtime

qsave(year_list, paste0("./Output data/sim_results_all_wealth_passed_no_transfer_", today_date, ".qs"))


## Reset partnered_status
partnered_status <- qread("./Input data/partnered_a.qs") %>% 
  rename(parent_age_grp = age_grp)

